package net.ouska.networking;

public class Eth implements Comparable<Eth> {

	private final String name;
	private final MAC mac;
	private final Host host;
	private boolean active;
	private IP ip;
	private IP mask;
	private boolean dhcp;
	private Network network;
	
	private boolean transmitting = false;
	
	public Eth(Host host, String name, MAC mac) {
		super();
		this.host = host;
		this.name = name;
		this.mac = mac;
	}
	
	public boolean isTransmitting() {
		return transmitting;
	}
	
	public void setTransmitting(boolean transmitting) {
		this.transmitting = transmitting;
	}
	
	public void reload() {
		if (active) {
			if (dhcp) {
				for (Eth eth : network.getEths().values()) {
					if (eth.isActive() && eth.getHost().getDhcp(network) != null) {
						try {
							Dhcp dhcp = eth.getHost().getDhcp(network);
							DhcpAssignment assignment = dhcp.assign(this.getMac());
							this.setIp(assignment.getIp());
							this.setMask(assignment.getMask());
							if (assignment.getGateway() != null) {
								this.getHost().setGateway(assignment.getGateway());	
							}
							if (assignment.getDns() != null) {
								this.getHost().setDns(assignment.getDns());	
							}
							return;
						} catch (Exception e) {
							host.getSystem().out("Host " + host.getName() + " " + this.getName() + " disabled - no DHCP configuration is available!");
							setActive(false);
						}
					}
				}
				host.getSystem().out("Host " + host.getName() + " " + this.getName() + " disbled - No DHCP server found in the local network!");
				setActive(false);
			}
		}
	}

	public void tryToEnable() {
		host.getSystem().out("Enabling " + this.getHost() /* + ":" + this.getName() +  "."*/);
		if (!this.isDhcp() && this.getIp() == null) {
			host.getSystem().out("Neither IP address is not defined or DHCP flag is set!");
		} else {
			this.setActive(true);
			if (this.isActive()) {
				host.getSystem().out(this.getHost() + /*":" + this.getName() + */ " enabled");	
			}
		}
	}
	
	@Override
	public String toString() {
		StringBuilder b = new StringBuilder();
		b.append(getName());
		b.append(" " + getMac());
		if (isDhcp()) {
			b.append("\t[DHCP] ");	
		}
		if (active) {
			b.append("\t" + getIp());
			b.append("\t" + getMask());
		} else {
			b.append("\tDISABLED");	
		}
		return b.toString();
	}

	public boolean isInMyNetwork(IP ipAddress) {
		return (ip.getA() & mask.getA()) == (ipAddress.getA() & mask.getA())
			&& (ip.getB() & mask.getB()) == (ipAddress.getB() & mask.getB())
			&& (ip.getC() & mask.getC()) == (ipAddress.getC() & mask.getC())
			&& (ip.getD() & mask.getD()) == (ipAddress.getD() & mask.getD());
	}
	
	public Network getNetwork() {
		return network;
	}

	public void setNetwork(Network network) {
		this.network = network;
	}

	public boolean isActive() {
		return active;
	}

	public void setActive(boolean active) {
		this.active = active;
		if (active) {
			reload();	
		}
	}

	public IP getIp() {
		return ip;
	}

	public void setIp(IP ip) {
		this.ip = ip;
	}

	public IP getMask() {
		return mask;
	}

	public void setMask(IP mask) {
		this.mask = mask;
	}

	public boolean isDhcp() {
		return dhcp;
	}

	public void setDhcp(boolean dhcp) {
		this.dhcp = dhcp;
	}

	public String getName() {
		return name;
	}

	public MAC getMac() {
		return mac;
	}

	public Host getHost() {
		return host;
	}

	@Override
	public int compareTo(Eth o) {
		return this.getName().compareTo(o.getName());
	}
}
