package net.ouska.networking;

import java.io.File;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.ouska.networking.command.Command;
import net.ouska.networking.gui.Okno;
import net.ouska.networking.hint.Hint;
import net.ouska.networking.hint.Param;

public class Sys {
	
	public Map<String, Host> hosts = new HashMap<String, Host>();
	public Map<String, Network> networks = new HashMap<String, Network>();
	public Host activeHost = null;
	public Console console;
	public Okno okno;
	public List<String> description = new ArrayList<String>() ;
	public boolean debugCommunication = false;
	
	public Sys(Console console, boolean readOnly) {
		this.console = console;
		okno = new Okno(this, readOnly);
		setActiveHost(null);
	}
	
	public void deleteAll() {
		hosts.clear();
		networks.clear();
		console.hostCommands.clear();
		activeHost = null;
		description.clear();
	}
	
	public void addHost(Host host) {
		hosts.put(host.getName(), host);
		okno.initHost(host);
		okno.onConfigurationChange();
	}

	public void removeHost(Host host) {
		for (Eth eth : host.getEths().values()) {
			if (eth.getNetwork() != null) {
				eth.getNetwork().deattach(eth);	
			}
		}
		hosts.remove(host.getName());
		okno.onConfigurationChange();
	}

	public void addNetwork(Network network) {
		networks.put(network.getName(), network);
		okno.initNetwork(network);
		okno.onConfigurationChange();
	}

	public void removeNetwork(Network network) {
		for (Host host : hosts.values()) {
			host.dhcp.remove(network);
		}
		Set<Eth> eths = new HashSet<>(network.getEths().values());
		for (Eth eth : eths) {
			network.deattach(eth);
		}
		networks.remove(network.getName());
		okno.onConfigurationChange();
	}

	public void setActiveHost(Host host) {
		this.activeHost = host;
		console.hostCommands.clear();
		if (this.activeHost != null) {
			console.root(activeHost.getName() + ":>");
			console.initHostCommands(activeHost);
			
		} else {
			console.root(":>");				
		}
		okno.onConfigurationChange();
	}
	
	public void attach(Eth eth, Network network) {
		for (Network n : networks.values()) {
			n.deattach(eth);
		}
		network.attach(eth);
		okno.onConfigurationChange();
	}
	
	public void onCommand(String line) {
		try {
			out(">> " + line);
			Options opt = new Options(line);
			if (!opt.has0() || opt.opt0().isEmpty()) {
				return;
			}

			Command command = null;
			
			if (activeHost != null) {
				command = console.hostCommands.get(opt.opt0());
			} else {
				command = console.commands.get(opt.opt0());	
			}
			
			if (command != null) {
				command.perform(opt);
			} else {
				out("Unknown command");		
			}	
			okno.onConfigurationChange();
		} catch (Exception e) {
			e.printStackTrace(System.err);
			if (e.getMessage() != null) {
				out(e.getMessage());	
			} else {
				StringWriter sw = new StringWriter();
				e.printStackTrace(new PrintWriter(sw));
				out(sw.toString());
			}
		}
	}		
	
	public void out(String str) {
		console.out(str);
	}

	public void debug(String str) {
		if (debugCommunication) {
			console.out("\t" + str);	
		}
	}

	public String hint(String line) {
		String completedLine = "";
		Hint hint = new Hint("");
		for (Command command : ((activeHost != null) ? console.hostCommands.values() : console.commands.values())) {
			hint.add(command.createHint());
		}
				
		List<String> options = new Options(line).options();
		if (line.endsWith(" ")) {
			options.add("");
		}
		
	   for (Iterator<String> it = options.iterator();  it.hasNext(); ) {
		   String option = it.next();
		   List<Hint> hints = hint.next(option);
		   if (!it.hasNext()) {
			   String auto = autocompletion(hints, option);
			   if (auto != null) {
				   completedLine = completedLine + ((completedLine.isEmpty()) ? "" : " ") + auto + " ";
				   return completedLine;
			   } else {
				   for (Hint h : hints) {
					   if (h instanceof Param) {
						   out(">>> [" +  h.getName() + "]");   
					   } else {
						   out(">>> " +  h.getName());
					   }
				   }
				   out(">>> ");
				   return null;
			   }
		   } else {
			   if (hints.size() == 1 && ( hints.get(0).getName().equalsIgnoreCase(option) || hints.get(0) instanceof Param)) {
				   hint = hints.get(0);
				   completedLine = completedLine + ((completedLine.isEmpty()) ? "" : " ") + option;
				   continue;
			   } else {
				   break;
			   }
		   }
	   }
	   
	   return null;
	}
	
	private String autocompletion(List<Hint> hints, String option) {
		Hint result = null;
		for (Hint hint : hints) {
			if (!(hint instanceof Param) || option.isEmpty()) {
				if (result != null) {
					return null;
				}
				result = hint;
			}
		}
		if (result != null && !(result instanceof Param)) {
			return result.getName();
		} else {
			return null;
		}
	}
	
	public static File getDefaultDirectory() {
		return new File(System.getProperty("user.home"));
	}
}
