package net.ouska.networking.command;

import java.awt.Dimension;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

import javax.swing.JFileChooser;

import net.ouska.networking.Dhcp;
import net.ouska.networking.Dns;
import net.ouska.networking.Email;
import net.ouska.networking.Eth;
import net.ouska.networking.Host;
import net.ouska.networking.Http;
import net.ouska.networking.IP;
import net.ouska.networking.MAC;
import net.ouska.networking.Network;
import net.ouska.networking.Options;
import net.ouska.networking.Sys;
import net.ouska.networking.hint.Hint;
import net.ouska.networking.hint.Keyword;

public class SystemCommandLoad extends SystemCommand {

	public SystemCommandLoad(Sys system, Sys template) {
		super(system, template, "load");
	}

	public void perform(Options op) {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setDialogTitle("Vyberte soubor");   
		int userSelection = fileChooser.showOpenDialog(system.console);
		if (userSelection != JFileChooser.APPROVE_OPTION) {
			return;
		}		
		File file = fileChooser.getSelectedFile();
		//File dir = Sys.getDefaultDirectory();;
		//File file = new File(dir, op.opt1() + ".ntw");

		if (!file.exists()) {
			out("Filename " + file.getAbsolutePath() + " not found!");
			return;
		}

		loadConfigurationFromFile(system, file);
		out("Network " + file.getAbsolutePath() + " loaded");
	}

	public static void loadConfigurationFromFile(Sys system, File file) {
		try (FileInputStream ifs = new FileInputStream(file)) {
			loadConfigurationFromFile(system, ifs);	
		} catch (IOException e) {
			throw new RuntimeException("Error loading file!", e);
		}
	}
	
	public static void loadConfigurationFromFile(Sys system, InputStream is) {
		Map<String, List<Map<String, String>>> content;
		try (Scanner scanner = new Scanner(is)) {
			content = readContent(scanner);	
		} catch (Exception e) {
			throw new RuntimeException("Error loading file!", e);
		}
		
		system.okno.setAutoPositioning(false);	
		
		system.deleteAll();
		for (Map<String, String> map : content.get("project")) {
			String line = valueString(map, "desc", false);
			if (line != null) {
				system.description.add(line);
			}
			int width = valueInt(map, "width", -1);
			int height = valueInt(map, "height", -1);
			if (width > -1 && height > -1) {
				system.okno.setSize(width, height);
				system.okno.setPreferredSize(new Dimension(width, height));
			}
		}
		for (Map<String, String> map : content.get("network")) {
			Network network = new Network(system, valueString(map, "name", true)); 
			network.x = valueInt(map, "x", -1);
			network.y = valueInt(map, "y", -1);
			system.addNetwork(network);
		}
		for (Map<String, String> map : content.get("host")) {
			Host host = new Host(system, valueString(map, "name", true)); 
			host.x = valueInt(map, "x", -1);
			host.y = valueInt(map, "y", -1);
			host.setGateway(valueIP(map, "gateway", false));
			host.setDns(valueIP(map, "dns", false));
			system.addHost(host);
		}
		for (Map<String, String> map : content.get("eth")) {
			String name = valueString(map, "name", true);
			Host host = valueHost(system, map, "host");
			MAC mac = valueMac(map, "mac", true);		
			Eth eth = new Eth(host, name, mac);
			host.addEth(eth);
			eth.setNetwork(valueNetwork(system, map, "network", false));
			eth.setIp(valueIP(map, "ip", false));
			eth.setMask(valueIP(map, "mask", false));
			eth.setActive(valueBoolean(map, "isActive"));
			eth.setDhcp(valueBoolean(map, "isDhcp"));
			if (eth.getNetwork() != null) {
				system.attach(eth, eth.getNetwork());	
			}
		}
		for (Map<String, String> map : content.get("dns")) {
			Host host = valueHost(system, map, "host");
			host.setDnsConfiguration(new Dns(host));
		}
		for (Map<String, String> map : content.get("dns-entry")) {
			Host host = valueHost(system, map, "host");
			host.getDnsConfiguration().getTable().put(
				valueString(map, "hostname", true), 
				valueIP(map, "ip", true));
		}
		for (Map<String, String> map : content.get("dhcp")) {
			Host host = valueHost(system, map, "host");
			Network network =  valueNetwork(system, map, "network", true);
			Dhcp dhcp = new Dhcp(host, 
				valueIP(map, "poolStart", true), 
				valueIP(map, "mask", true), 
				valueIP(map, "gateway", false), 
				valueIP(map, "dns", false));
			dhcp.setCounter(valueInt(map, "counter", 0));
			host.setDhcp(network, dhcp);
		}
		for (Map<String, String> map : content.get("dhcp-entry")) {
			Host host = valueHost(system, map, "host");
			Network network =  valueNetwork(system, map, "network", true);
			Dhcp dhcp = host.getDhcp(network);
			dhcp.getTable().put(valueMac(map, "mac", true), valueIP(map, "ip", true));
		}
		for (Map<String, String> map : content.get("http")) {
			Host host = valueHost(system, map, "host");
			host.setHttp(new Http(host));
		}
		for (Map<String, String> map : content.get("http-entry")) {
			Host host = valueHost(system, map, "host");
			Http http = host.getHttp();
			http.addFile(valueString(map, "filename", true), valueString(map, "content", false));
		}
		for (Map<String, String> map : content.get("email")) {
			Host host = valueHost(system, map, "host");
			host.setEmail(new Email(host));
		}
		for (Map<String, String> map : content.get("email-entry")) {
			Host host = valueHost(system, map, "host");
			Email email = host.getEmail();
			email.addMessage(valueString(map, "recipient", true), valueString(map, "message", false));
		}
	}
	
	private static String valueString(Map<String, String> map, String name, boolean mandatory) {
		String value = map.get(name);
		if (mandatory && value == null) {
			throw new RuntimeException("Mandatory value " + name + " missing!");
		}
		return value;
	}

	private static boolean valueBoolean(Map<String, String> map, String name) {
		String value = map.get(name);
		return value != null && Boolean.valueOf(value);
	}

	private static IP valueIP(Map<String, String> map, String name, boolean mandatory) {
		String value = valueString(map, name, mandatory);
		if (value == null) {
			return null;
		} else {
			return new IP(value);	
		}
	}

	private static Host valueHost(Sys system, Map<String, String> map, String name) {
		String value = valueString(map, name, true);
		Host host = system.hosts.get(value);
		if (host == null) {
			throw new RuntimeException("Unknown host " + value + " missing!");
		}
		return host;	
	}

	private static Network valueNetwork(Sys system, Map<String, String> map, String name, boolean mandatory) {
		String value = valueString(map, name, mandatory);
		if (value == null) {
			return null;
		}
		return system.networks.get(value);
	}

	private static MAC valueMac(Map<String, String> map, String name, boolean mandatory) {
		return new MAC(valueString(map, name, mandatory));	
	}

	private static int valueInt(Map<String, String> map, String name, int defaultValue) {
		String value = map.get(name);
		int result = defaultValue;
		if (value != null) {
			result = Integer.valueOf(map.get(name));
		}
		return result;
	}
	
	public static Map<String, List<Map<String, String>>> readContent(Scanner scanner) {
		Map<String, List<Map<String, String>>> content = new HashMap<String, List<Map<String, String>>>();
		content.put("project", new ArrayList<Map<String,String>>());
		content.put("host", new ArrayList<Map<String,String>>());
		content.put("network", new ArrayList<Map<String,String>>());
		content.put("eth", new ArrayList<Map<String,String>>());
		content.put("dns", new ArrayList<Map<String,String>>());
		content.put("dns-entry", new ArrayList<Map<String,String>>());
		content.put("dhcp", new ArrayList<Map<String,String>>());
		content.put("dhcp-entry", new ArrayList<Map<String,String>>());
		content.put("email", new ArrayList<Map<String,String>>());
		content.put("email-entry", new ArrayList<Map<String,String>>());
		content.put("http", new ArrayList<Map<String,String>>());
		content.put("http-entry", new ArrayList<Map<String,String>>());
		content.put("http-entry", new ArrayList<Map<String,String>>());
		
		while (scanner.hasNextLine()) {
			String line = scanner.nextLine();
			String[] couples = line.split(";");
			String klic = couples[0].trim().toLowerCase();
			
			List<Map<String, String>> list = content.get(klic);
			if (list == null) {
				list = new ArrayList<Map<String,String>>();
				content.put(klic, list);
			}
			
			Map<String, String> mapa = new HashMap<String, String>();
			list.add(mapa);
			for (int i = 1; i < couples.length; i++) {
				String[] pair = couples[i].split("=");
				if (pair.length > 1) {
					mapa.put(pair[0].trim(), pair[1].trim());	
				}
			}
		}
		return content;
	}
	
	@Override
	public Hint createHint() {
		Hint root = super.createHint();
		/*
		File home = Sys.getDefaultDirectory();
		for (File file : home.listFiles()) {
			if (file.isFile() && file.getName().endsWith(".ntw")) {
				root.add(new Keyword(file.getName().substring(0, file.getName().length() - 4)));				
			}
		} 
		*/
		return root;
	}
	
}
