package net.ouska.networking.gui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.RenderingHints;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.imageio.ImageIO;
import javax.swing.JPanel;

import net.ouska.networking.Eth;
import net.ouska.networking.Host;
import net.ouska.networking.Network;
import net.ouska.networking.Sys;

public class Canvas extends JPanel implements MouseListener, MouseMotionListener {

	public static final int SIZE_HOST = 70;
	public static final int NETWORK_WIDTH = 75;
	public static final int NETWORK_HEIGHT = 33;
	
	private Sys system;
	private Host activeHost;
	private Network activeNetwork;
	private int clickX;
	private int clickY;
	private boolean autoPositioning = true;
	private final boolean readOnly;
	
	private BufferedImage networkImage;
	private BufferedImage hostImage;
	
	
    public Canvas(Sys system, boolean readOnly) {
		super();
		this.system = system;
		this.readOnly = readOnly;
		if (!readOnly) {
			addMouseMotionListener(this);
			addMouseListener(this);
		}
		
		try {
	    	//File file = new File(ImageAccessor.class.getResource("sw.png").getFile());
			BufferedImage in = ImageIO.read(ImageAccessor.class.getResource("sw.png").openStream());
			networkImage = new BufferedImage(in.getWidth(), in.getHeight(), BufferedImage.TYPE_INT_ARGB);
			Graphics2D gg = networkImage.createGraphics();
			gg.drawImage(in, 0, 0, null);
			gg.dispose();
			
    		//file = new File(ImageAccessor.class.getResource("pc.png").getFile());
    		in = ImageIO.read(ImageAccessor.class.getResource("pc.png").openStream());
    		hostImage = new BufferedImage(in.getWidth(), in.getHeight(), BufferedImage.TYPE_INT_ARGB);
    		gg = hostImage.createGraphics();
    		gg.drawImage(in, 0, 0, null);
    		gg.dispose();
    	} catch (Exception e) {
			e.printStackTrace();
		} 		
	}

    public void setAutoPositioning(boolean autoPositioning) {
    	this.autoPositioning = autoPositioning;
    }

    private void autoComputePositions(Graphics2D g) {
    	Grid grid = new Grid(system, this.getWidth(), this.getHeight());
    	//grid.draw(g);
    	List<Network> networks = new ArrayList<Network>(system.networks.values());
    	Collections.sort(networks);
    	for (int i = 0; i < networks.size(); i++) {
    		Network network = networks.get(i);
    		network.x = grid.getCenterX(2*i + 1) - NETWORK_WIDTH/2;
    		network.y = grid.getCenterY(1) - NETWORK_HEIGHT/2;    		
		}
    	Set<Host> alreadyPlacedHosts = new HashSet<Host>();
    	for (int i = 0; i < networks.size(); i++) {
    		Network network = networks.get(i);
    		Host hostGateway = findHostBetweenNetworks(networks, i);
    		if (hostGateway != null) {
    			if (!alreadyPlacedHosts.contains(hostGateway)) {
        			hostGateway.x = grid.getCenterX(2*i + 2) - SIZE_HOST/2;
        			hostGateway.y = grid.getCenterY(1) - SIZE_HOST/2;
        			alreadyPlacedHosts.add(hostGateway);
    			}
    			Set<Host> hosts = new HashSet<Host>();
    			for (Eth eth : network.getEths().values()) {
    				hosts.add(eth.getHost());
    			}
    			hosts.remove(hostGateway);
    	    	List<Host> hostList = new ArrayList<Host>(hosts);
    	    	Collections.sort(hostList);
    	    	hostList.removeAll(alreadyPlacedHosts);
    	    	int centerX = grid.getCenterX(2*i + 1);
    	    	int centerY = grid.getCenterY(1);
	    		int rx = grid.getCellWidth();
	    		int ry = grid.getCellHeight();
	    		double uhel = (centerX > hostGateway.x) ?  Math.PI : 0;
	    		//g.drawOval(centerX - rx, centerY - ry, 2*rx, 2*ry);
	    		//System.out.println("Host list size: " + hostList.size() + " rx=" + rx + " ry=" + ry + " centerX=" + centerX + " centerY=" + centerY );
    	    	for (int j = 0; j < hostList.size(); j++) {
    	    		uhel = uhel + (2*Math.PI) / (hostList.size() + 1);
    	    		Host host = hostList.get(j);
    	    		host.x = (int) (centerX + Math.cos(uhel) * rx - SIZE_HOST/2);
    	    		host.y = (int) (centerY + Math.sin(uhel) * ry - SIZE_HOST/2);
    	    		alreadyPlacedHosts.add(host);
    	    		//System.out.println("host " + host.getName() + " uhel: " + uhel + " x=" + host.x  + " y=" + host.y);
				}
    		}
		} 
    }
    
    public Host findHostBetweenNetworks(List<Network> list, int position) {
    	List<Host> hosts = new ArrayList<Host>(system.hosts.values());
    	Collections.sort(hosts);
    	Host result = null;
    	Network network1 = list.get(position);
    	Network network2 = null;
    	if (position + 1 < list.size()) {
    		network2 = list.get(position + 1);
    		
    	}
		for (Host host : hosts) {
			Set<Network> networks = new HashSet<Network>(); 
			for (Eth eth : host.getEths().values()) {
				networks.add(eth.getNetwork());
			}
			if (networks.contains(network1)
				&& networks.contains(network2)) {
				result = host;
				break;
			} else if (networks.contains(network1)) {
				result = host;
			}
		}
    	return result;
    }
    
	@Override
    public void paintComponent(Graphics gs) {
    	super.paintComponent(gs);
    	Graphics2D g = (Graphics2D) gs;
    	g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
    	if (autoPositioning) {
    		autoComputePositions(g);	
    	}
    	
    	for (int i = 0; i < system.description.size(); i++) {
    		g.drawString(system.description.get(i), 10, 20 + i*15);
		}
    	
    	for (Host host : system.hosts.values()) {
    		for (Eth eth : host.getEths().values()) {
    			Network network = eth.getNetwork();
    			if (network != null) {
        			
    				if (!eth.isActive()) {
        				g.setColor(Color.LIGHT_GRAY);	
        			} else {
        				if (eth.isTransmitting()) {
        					g.setColor(Color.RED);
        					//g.setStroke(null);
        					g.setStroke(new BasicStroke(10));
        				} else {
        					g.setColor(Color.BLACK);
        					g.setStroke(new BasicStroke(1));
        				}
        			}
        			g.drawLine(host.x + SIZE_HOST/2, host.y + SIZE_HOST/2, network.x + NETWORK_WIDTH/2, network.y + NETWORK_HEIGHT/2);
        			g.setStroke(new BasicStroke(1));
        			g.setColor(Color.BLACK);

        			int hostCenterX = host.x + SIZE_HOST/2;
        			int hostCenterY = host.y + SIZE_HOST/2;
        			int networkCenterX = network.x + NETWORK_WIDTH/2;
        			int networkCenterY = network.y + NETWORK_HEIGHT/2;
        			int x = (networkCenterX - hostCenterX) / 2; 
        			int y = (networkCenterY - hostCenterY) / 2;
        			
        			List<String> list = new ArrayList<String>();
        			list.add(eth.getName() + "  " + eth.getMac());
        			if (eth.getIp() != null) {
        				list.add(eth.getIp().toString());	
        			}
        			if (eth.getMask() != null) {
        				list.add(eth.getMask().toString());	
        			}
        			if (eth.getHost().getGateway() != null) {
        				list.add("GW: " + eth.getHost().getGateway().toString());	
        			}
        			printBox(g, list, hostCenterX+x, hostCenterY+y);
    			} 
			} 
		}
		
    	for (Host host : system.hosts.values()) {
    		paint(g, host);
		}
    	
    	for (Network network : system.networks.values()) {
    		paint(g, network);
		}
    }			

    private void printBox(Graphics2D g, List<String> texts, int centerX, int centerY) {
		int width = 0;
		for (String text : texts) {
			int w = g.getFontMetrics().stringWidth(text);
			if (w > width) {
				width = w;
			}
		}
		int lineHeight = 10;
		int height = texts.size() * lineHeight;
		int padding = 5;
		g.setColor(Color.WHITE);
        g.fillRect(centerX - width/2 - padding, centerY - height/2 - padding, width + 2*padding, height + 2*padding);
        g.setColor(Color.BLACK);
        g.drawRect(centerX - width/2 - padding, centerY - height/2 - padding, width + 2*padding, height + 2*padding);
        for (int i = 0; i < texts.size(); i++) {
        	int w = g.getFontMetrics().stringWidth(texts.get(i));
        	g.drawString(texts.get(i), centerX - w/2, centerY - height/2 + (i+1)*lineHeight);	
		}
    }
	
    public void paint(Graphics2D g, Host host) {
    	g.drawImage(hostImage, host.x, host.y, null);
    	int i=0;
    	centerText(g, host.getName(), host, i++);
    	/*
    	if (host.getDns() != null) {
    		centerText(g, "DNS:" + host.getDns(), host, i++);	
    	}
    	if (host.getGateway() != null) {
    		centerText(g, "GW:" + host.getGateway(), host, i++);	
    	}
    	 */
    	
    	if (host == system.activeHost) {
    		g.setColor(Color.WHITE);
    		g.fillRect(host.x + 8, host.y + 5, SIZE_HOST - 16, SIZE_HOST - 38);	
    	}    	
    	g.setColor(host == system.activeHost ? Color.BLACK : Color.WHITE);
    	StringBuilder b = new StringBuilder();
    	if (!host.getDhcps().isEmpty()) {
    		b.append(" DHCP ");	
    	}
    	if (host.getDnsConfiguration() != null) {
    		b.append(" DNS ");	
    	}
    	if (host.getEmail() != null) {
    		b.append(" EMAIL ");	
    	}
    	if (host.getHttp() != null) {
    		b.append(" HTTP ");	
    	}

    	if (b.length() > 0) {
    		int width = g.getFontMetrics().stringWidth(b.toString());
    		g.drawString(b.toString(), host.x + SIZE_HOST/2 - width/2, host.y + SIZE_HOST/3);
    	}
    	
    	g.setColor(Color.BLACK);
    }			

    private void centerText(Graphics2D g, String text, Host host, int line) {
    	int width = g.getFontMetrics().stringWidth(text);
    	g.drawString(text, host.x + SIZE_HOST/2 - width/2, host.y + SIZE_HOST + SIZE_HOST/5 + line*12);
    }
    
    public void paint(Graphics2D g, Network network) {
    	g.drawImage(networkImage, network.x, network.y, null);
    	List<String> list = new ArrayList<>();
    	list.add(network.getName());
    	printBox(g, list, network.x + NETWORK_WIDTH/2, network.y + NETWORK_HEIGHT + NETWORK_HEIGHT/3);
    }			

    private Host findHost(int x, int y) {
    	for (Host host : system.hosts.values()) {
    		if (x >= host.x && x <= host.x + SIZE_HOST && y >= host.y && y <= host.y + SIZE_HOST) {
    			return host;
    		}
		}
    	return null;
    }

    private Network findNetwork(int x, int y) {
    	for (Network network : system.networks.values()) {
    		if (x >= network.x && x <= network.x + NETWORK_WIDTH && y >= network.y && y <= network.y + NETWORK_HEIGHT) {
    			return network;
    		}
		}
    	return null;
    }

	@Override
	public void mouseDragged(MouseEvent e) {
		if (activeHost != null) {
			int x = e.getX() - clickX;
			int y = e.getY() - clickY;
			activeHost.x = x - x % 10;
			activeHost.y = y - y % 10;
			setAutoPositioning(false);
		}
		if (activeNetwork != null) {
			/*
			for (Eth eth : activeNetwork.getEths().values()) {
				Host host = eth.getHost();
				int deltaX = activeNetwork.x - (e.getX() - clickX);
				int deltaY = activeNetwork.y - (e.getY() - clickY);
				host.x = host.x - deltaX;
				host.y = host.y - deltaY;
			}
			*/
			int x = e.getX() - clickX;
			int y = e.getY() - clickY;
			activeNetwork.x = x - x % 10;
			activeNetwork.y = y - y % 10;
			setAutoPositioning(false);
		}
		this.repaint();
	}

	@Override
	public void mouseMoved(MouseEvent e) {
		//System.out.println("yyy");
		//System.out.println("xxx " + e.getX());
	}

	@Override
	public void mouseClicked(MouseEvent e) {
		Host host = findHost(e.getX(), e.getY());
		if (host != null) {
			if (system.activeHost == null || system.activeHost != host) {
				if (system.activeHost != null) {
					system.out("Disconnected from " + system.activeHost.getName() + ".");
				}
				system.setActiveHost(host);
				system.out("Connected to " + host.getName() + ".");
				system.console.setVisible(true);
			} else if (system.activeHost != null) {
				system.setActiveHost(null);
				system.out("Disconnected from " + host.getName() + ".");
				system.console.setVisible(true);
			}
		}
	}

	@Override
	public void mousePressed(MouseEvent e) {
		Host host = findHost(e.getX(), e.getY());
		Network network = findNetwork(e.getX(), e.getY());
		activeHost = null;
		activeNetwork = null;
		if (host != null) {
			activeHost = host;
			clickX = e.getX() - host.x;
			clickY = e.getY() - host.y;
		} else if (network != null) {
			activeNetwork = network;
			clickX = e.getX() - network.x;
			clickY = e.getY() - network.y;
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		activeHost = null;
		activeNetwork = null;
	}

	@Override
	public void mouseEntered(MouseEvent e) {
		// TODO Auto-generated method stub
		
	}

	@Override
	public void mouseExited(MouseEvent e) {
		// TODO Auto-generated method stub
		
	}    
}
